(* A definition of streams in Coq *)
(* MP - MPRI - cours notes - Oct. 2021 *)
From Coq Require Import Extraction.
Set Implicit Arguments.
Require Import Coq.Init.Datatypes.

Require Import List.
Import ListNotations.

CoInductive Stream ( A : Type ) : Type :=
| Eps : Stream A -> Stream A
| Cons : A -> Stream A -> Stream A.

(* bottom/epsilon element on streams *)
CoFixpoint bot (A : Type) : Stream A := Eps (bot A).

(* The case operator was used by Ch. Paulin in his paper of 2009] *)
(* not sure it is really useful *)
CoFixpoint case {A B : Type}
           (f : A -> Stream A -> Stream B) : Stream A -> Stream B :=
  fun s =>
    match s with
    | Eps s => Eps (case f s)
    | Cons a s => f a s
    end.

CoFixpoint const {A : Type} (a : A) : Stream A := Cons a (const a).

(* the tail of a stream *)
Definition next {A : Type} : Stream A -> Stream A := case (fun a s => s).

Definition fby {A : Type} : Stream A -> Stream A -> Stream A :=
  fun X Y => case (fun a X => Cons a (next Y)) X.

(* Unit delay *)
Definition pre {A : Type} : A -> Stream A -> Stream A :=
  fun v xs => Cons v xs.

CoFixpoint extend {A B : Type } :
  Stream (A -> B) -> Stream A -> Stream B :=
  fun fs xs =>
  match fs, xs with
  | (Eps fs), (Eps xs) =>
      (* This case is subsumed by the last two. Yet we add it *)
      (* purposely because only the last two reveil that inputs *)
      (* are (implicitely) buffered when one is present only *)
      Eps (extend fs xs)
  | (Cons f fs), (Cons x xs) => Cons (f x) (extend fs xs)
  | (Eps fs), _ => Eps (extend fs xs)
  | _, (Eps xs) => Eps (extend fs xs)
end.

CoFixpoint when {A : Type } :
  Stream A -> Stream bool -> Stream A :=
  fun xs cs =>
    match xs, cs with
    | Eps(xs), Eps(cs) =>
        Eps(when xs cs)
    | (Cons x xs), (Cons c cs) =>
        if c then Cons x (when xs cs) else Eps(when xs cs)
    | Eps(xs), _ => Eps(when xs cs)
    | _, Eps(cs) => Eps(when xs cs)
    end.

CoFixpoint merge {A : Type } :
  Stream bool -> Stream A -> Stream A -> Stream A := 
    fun cs xs ys =>
      match cs, xs, ys with
      | Eps(cs), Eps(xs), Eps(ys) => Eps(merge cs xs ys)
      | (Cons true cs), (Cons x xs), _ =>
          Cons x (merge cs xs ys)
      | (Cons false cs), _, (Cons y ys) =>
          Cons y (merge cs xs ys)
      | Eps(cs), _, _ =>
          Eps(merge cs xs ys)
      | Cons _ _, Eps(xs), _ => Eps(merge cs xs ys)
      | Cons _ _, _, Eps(ys) => Eps(merge cs xs ys)
      end.

(* Constructive fixpoint operation *)
(* As we cannot define lim_{n->infty}(f^n(eps)), we use *)
(* a diagonal argument. Intuitively, build the stream such that element *)
(* of index j is the jth-element of f^n(eps) with j <= n *)
(* n defines the diagonal *)
(* This fix point expect [f] to be monotonous/continuous in order to have *)
(* [lfp(f) <= f(lfp(f))] and [f(lfp(f)) <= lfp(f)] with [<=] the Kahn order *)
Fixpoint ask_option
         { A : Type } (xs : Stream A) (j : nat) (n : nat) : option A :=
  (* return the j-ith value of [xs], with j<=n if it is present; 
   *- None otherwise *)
  match n with
  | 0 =>  None
  | S n =>
      match xs with
      | Eps(xs) => ask_option xs j n
      | Cons x xs =>
          match j with
          | 0 => Some(x)
          | S j => ask_option xs j n
          end
      end
  end.

CoFixpoint lfpaux { A : Type } (f : Stream A -> Stream A) (v : Stream A)
           (j : nat) (n : nat) : Stream A :=
  let v := f v in
  let vn := ask_option v j n in
  match vn with
  | None => Eps(lfpaux f v j (n+1))
  | Some(vn) => Cons vn (lfpaux f v (j+1) (n+1))
  end.

Definition lfp { A : Type } (f : Stream A -> Stream A) := lfpaux f (bot A) 0 0.

(* The following is not valid *)
(*
CoFixpoint lfp2 { A : Type } (f : Stream A -> Stream A) :=
  Eps(f(lfp2 f)).
*)
                                                                 
(* stream of pairs and pairs of streams *)
Definition pair { A B : Type } : Stream A -> Stream B -> Stream (A * B) :=
  fun xs ys =>
    extend (extend (const (fun x y => (x, y))) xs) ys.

Definition fst { A B : Type } : Stream (A * B) -> Stream A :=
  fun xs => extend (const (fun x => fst x)) xs.

Definition snd { A B : Type } : Stream (A * B) -> Stream B :=
  fun xs => extend (const (fun x => snd x)) xs.

Definition notl : Stream bool ->  Stream bool :=
  fun xs => extend (const (fun x => negb x)) xs.

Definition andl : Stream bool -> Stream bool -> Stream bool :=
  fun xs ys => extend (extend (const (fun x y => andb x y)) xs) ys.

Definition plusl : Stream nat -> Stream nat -> Stream nat :=
  fun xs ys => extend (extend (const (fun x y => x + y)) xs) ys.

Recursive Extraction fby.
(* Recursive Extraction "machin.ml" fby *)

Definition half0 :=
  let f := fun x => notl (pre false x) in
  lfp f.

Definition half1 :=
  let f := fun x => (pre true (notl x)) in
  lfp f.

(* 2. nat *)
Definition pre_incr v x := fby v (plusl x (const 1)).
               
Definition from v :=
  let f := pre_incr v in
  lfp f.
  
(* 2. unbounded memory *)
Definition unbounded :=
  let t := const true in
  let h := half0 in
  andl t (when t h).

 
(* 3. Causality loop *)
Definition deadlock1 { A : Type } : Stream A :=
  let deadlock := lfp (fun x => x) in
  deadlock.

Definition deadlock2 :=
  let deadlock := lfp (fun x => plusl x (const 1)) in
  deadlock.

(* 3. N-synchrone *)
Definition n_synchrone :=
  let t := from (const 0) in
  let h := half0 in
  plusl (when t (notl h)) (when t h).
  
Fixpoint list_of { A : Type } (n : nat) (xs : Stream A) :=
  match n with
  | 0 => []
  | S n =>
      match xs with
      | Eps(xs) => None :: (list_of n xs)
      | Cons x xs => Some(x) :: (list_of n xs)
      end
  end.

Eval vm_compute in (list_of 100 (deadlock1)).

Eval vm_compute in (list_of 100 (n_synchrone)).

Eval vm_compute in (list_of 100 (unbounded)).

Eval vm_compute in (list_of 10 (from (const 0))).

Eval vm_compute in (list_of 100 (half0)).
Eval vm_compute in (list_of 100 (half1)).
