(* An encoding of the Lustre/Lucid Synchrone data-flow operators in OCaml *)
(* MP - oct. 2021 - MPRI - Course notes *)

(* e ::= e fby e | pre v e | extend e e | e when e | merge e e e
      | fun x -> e | e (e) |  fix e | reset e every e | do e until e then e
*)

(* A stream is a lazy data-structure
 *- (1) Contrary to definitions in [kahn.ml] we add a way to represent
 *- the bottom elements over streams.
 *- The intuition is this: 
 *- between two successive elements of a streams,
 *- it is always possible to insert a silent element Eps. 
 *- The bottom element, usually noted [eps] 
 *- is represented explicitly as the solution of the equation 
 *- [eps = Eps(eps)].
 *- (2) We present an encoding in OCaml and leave as a (very interesting)
 *- exercise its implementation in Coq.
 *- the idea exploited here where streams are complemented in order to
 *- be able to represent bottom explicitly is due to Ch. Paulin 
 *- [A constructive denotational semantics for Kahn networks in Coq, 2009]
 *- (3) The fixpoint construction we implement in OCaml is very simple; yet
 *- it cannot be defined in Coq because the guard condition of Coq is not
 *- expressive enough and does not pass abstraction bounderies *)

type 'a lazy_stream = 'a stream Lazy.t
and 'a stream =
  | Eps :
      'a lazy_stream -> 'a stream
  | Cons :
      'a * 'a lazy_stream -> 'a stream

let rec eps = lazy (Eps(eps))
            
(* Constant generation *)
let rec const: 'a -> 'a lazy_stream = fun v -> lazy (Cons(v, const v))

let rec extend : ('a -> 'b) lazy_stream -> 'a lazy_stream -> 'b lazy_stream =
  fun fs xs ->
  lazy
    (match Lazy.force fs, Lazy.force xs with
     | Eps(fs), Eps(xs) ->
        (* This case is subsumed by the last two. Yet we add it *)
        (* purposely because only the last two reveil that inputs *)
        (* are (implicitely) buffered when one is present only *)
        Eps(extend fs xs)
     | Cons(f, fs), Cons(x, xs) -> Cons(f x, extend fs xs)
     | Eps(fs), _ -> Eps(extend fs xs)
     | _, Eps(xs) -> Eps(extend fs xs))
                   
(* Unit delay *)
let pre : 'a -> 'a lazy_stream -> 'a lazy_stream =
  fun v xs -> lazy (Cons(v, xs))

(* Initialized delay. The fby operator comes from the old Lucid language *)
let rec fby : 'a lazy_stream -> 'a lazy_stream -> 'a lazy_stream =
  fun xs ys ->
  match Lazy.force xs with
  | Eps(xs) -> lazy (Eps(fby xs ys))
  | Cons(v, _) -> pre v ys

(* stream of pairs and pairs of streams *)
let pair (xs, ys) =
  extend (extend (const (fun x y -> (x, y))) xs) ys

let fst xs = extend (const (fun (x, _) -> x)) xs
let snd xs = extend (const (fun (_, y) -> y)) xs

let notl xs = extend (const (fun x -> not x)) xs
let andl xs ys = extend (extend (const (fun x y -> x && y)) xs) ys

let plusl xs ys = extend (extend (const (fun x y -> x + y)) xs) ys

let rec whenc xs cs =
  lazy
    (match Lazy.force xs, Lazy.force cs with
     | Eps(xs), Eps(cs) ->
        Eps(whenc xs cs)
     | Cons(x, xs), Cons(c, cs) ->
        if c then Cons(x, whenc xs cs) else Eps(whenc xs cs)
     | Eps(xs), _ -> Eps(whenc xs cs)
     | _, Eps(cs) -> Eps(whenc xs cs))

let rec merge cs xs ys =
  lazy
    (match Lazy.force cs, Lazy.force xs, Lazy.force ys with
     | Eps(cs), Eps(xs), Eps(ys) -> Eps(merge cs xs ys)
     | Cons(true, cs), Cons(x, xs), _ ->
        Cons(x, merge cs xs ys)
     | Cons(false, cs), _, Cons(y, ys) ->
        Cons(y, merge cs xs ys)
     | Eps(cs), _, _ ->
        Eps(merge cs xs ys)
     | Cons _, Eps(xs), _ -> Eps(merge cs xs ys)
     | Cons _, _, Eps(ys) -> Eps(merge cs xs ys))
                                   
                     
(* Least fix-point operator for streams *)
let fix f =
  let rec y = lazy (Eps(f y)) in
  y

let fix2 f = 
  let xy = fix (fun xy -> pair (f (fst xy, snd xy))) in
  fst xy, snd xy

(* The two definitions above cannot be defined in Coq. [fix f] 
*- cannot be given a constructive definition, that is, given any stream 
*- function f, computes its least fix point. E.g., there is no answer 
*- for [fix id] where id x = x. In the above definition, the execution would
*- stuck *)

(* We present now a constructive definition of the least fix-point
 *- it builds on a diagonal argument. The intuition is this:
*- let bounded_fix(f)(n) = f^n(eps);
*- let nth_option(s)(n) = None if the n-th element of s is Eps;
*-                      = f^n(eps) otherwise;
*- return the stream s such that s(n) = nth_option(bounded_fix(f)(n))(n).
*- this function can be defined constructively (using cofixpoint in Coq)
*- and verifies the guard conditions of Coq. 
*- Yet, it is extremely inefficient! Can you do better? *)

let rec nth_option xs n =
  match Lazy.force xs with
  | Cons(x, xs) -> if n = 0 then Some(x) else nth_option xs (n-1)
  | Eps(xs) -> if n = 0 then None else nth_option xs (n-1)

(* Bounded iteration *)
let blfp f n =
  let rec bounded_rec n v =
    let v = f v in
    if n = 0 then v
    else bounded_rec (n-1) v in
  bounded_rec n eps

(* Least fix-point *)
let lfp f =
  let rec lfp v n =
    let v = f v in
    let vn = nth_option v n in
    lazy
      (match vn with
       | None -> Eps(lfp v (n+1))
       | Some(vn) -> Cons(vn, lfp v (n+1))) in
  lfp eps 0

(*- [lfp f] expects [f] to be length preserving. *)
(*- we do a more general version now *)
let rec ask_option xs j n =
  (* return the j-ith value of [xs], with j<=n if it is present; 
   *- None otherwise *)
  if n = 0 then None
  else
    match Lazy.force xs with
    | Eps(xs) -> ask_option xs j (n-1)
    | Cons(x, xs) -> if j = 0 then Some(x) else ask_option xs (j-1) (n-1)

(* Least fix-point *)
let lfp f =
  let rec lfp v j n =
    let v = f v in
    let vn = ask_option v j n in
    lazy
      (match vn with
       | None -> Eps(lfp v j (n+1))
       | Some(vn) -> Cons(vn, lfp v (j+1) (n+1))) in
  lfp eps 0 0

(* two streams are said to be equivalent iff they have *)
(* the same elements --- that is, up to the erasure of Eps elements *)
(* show that if f is monotonous, lfp(f) <=_prefix f(lfp(f)) *)
(* and iff f is continuous, f(lfp(f)) <=_prefix lfp(f) *)
  
(* exemples *)
(* 1. half *)
let half () =
  let f = fun x -> notl (pre false x) in
  lfp f

let half () =
  let f = fun x -> (pre true (notl x)) in
  lfp f

(* 2. nat *)
let pre_incr v x = pre v (plusl x (const 1))
               
let from v =
  let f = pre_incr v in
  lfp f

(* 2'. nat using a constructive fixpoint *)
let from v =
  let f = fun x -> pre v (plusl x (const 1)) in
  lfp f
  
(* 2. unbounded memory *)
let unbounded () =
  let t = const true in
  let h = half () in
  andl t (whenc t h)

 
(* 3. Causality loop *)
let deadlock1 () =
  let deadlock = fix (fun x -> x) in
  deadlock

let deadlock2 () =
  let deadlock = fix (fun x -> plusl x (const 1)) in
  deadlock

(* 3. N-synchrone *)
let n_synchrone () =
  let t = from 0 in
  let h = half () in
  plusl (whenc t (notl h)) (whenc t h)
  
let rec list_of n xs =
  if n = 0 then []
  else match Lazy.force xs with
       | Eps(xs) -> None :: (list_of (n-1) xs)
       | Cons(x, xs) -> Some(x) :: (list_of (n-1) xs)

let rec out print n xs =
  if n = 0 then ()
  else match Lazy.force xs with
       | Eps(xs) -> print_string ". "; print_newline (); out print (n-1) xs
       | Cons(x, xs) -> print x; out print (n-1) xs

let print x = print_int x; print_newline ()

let print_bool x =
  print_string (if x then "true" else "false"); print_newline ();;

out print 1000 (lfp (pre_incr 0))
(* out print_bool 100000 (unbounded ()) *)
