(* An encoding of the Lustre/Lucid Synchrone data-flow operators in OCaml *)
(* MP - oct. 2021 - MPRI - Course notes *)

(* e ::= e fby e | pre v e | extend e e | e when e | merge e e e
      | fun x -> e | e (e) |  fix e | reset e every e | do e until e then e
*)

(* A stream is a lazy data-structure *)
type 'a lazy_stream = 'a stream Lazy.t
and 'a stream = | Cons : 'a * 'a lazy_stream -> 'a stream
                       
type ('a, 'b) system = 'a lazy_stream -> 'b lazy_stream
                     
(* Constant generation *)
let rec const : 'a -> 'a lazy_stream = fun v -> lazy (Cons(v, const v))

(* Combinatorial function lifting *)
let rec extend :
          ('a -> 'b) lazy_stream -> 'a lazy_stream -> 'b lazy_stream =
  fun fs xs ->
  lazy
    (match Lazy.force fs, Lazy.force xs with
     | Cons(f, fs), Cons(x, xs) ->
        Cons(f x, extend fs xs))
                     
(* Unit delay *)
let pre : 'a -> 'a lazy_stream -> 'a lazy_stream =
  fun v xs -> lazy (Cons(v, xs))

(* Initialized delay. The fby operator comes from the old Lucid language *)
let rec fby : 'a lazy_stream -> 'a lazy_stream -> 'a lazy_stream =
  fun xs ys ->
  match Lazy.force xs with
  | Cons(v, _) -> pre v ys

(* stream of pairs and pairs of streams *)
let pair : 'a lazy_stream * 'b lazy_stream -> ('a * 'b) lazy_stream =
  fun (xs, ys) ->
  extend (extend (const (fun x y -> (x, y))) xs) ys

let plus1 xs = extend (const (fun x -> x + 1)) xs
             
let fst xs = extend (const (fun (x, _) -> x)) xs
let snd xs = extend (const (fun (_, y) -> y)) xs

let notl xs = extend (const (fun x -> not x)) xs
let andl xs ys = extend (extend (const (fun x y -> x && y)) xs) ys

let eql xs ys =
  extend (extend (const (fun x y -> x = y)) xs) ys

let mux cs xs ys =
  extend (extend (extend
                    (const (fun c x y -> if c then x else y)) cs) xs) ys

let plusl xs ys = extend (extend (const (fun x y -> x + y)) xs) ys

(* Initialization. [e1 -> e2] written here [e1 --> e2] *)
(* [e1 -> e2] is a shortcut for [if (true fby false) then e1 else e2] *)
let (-->) : 'a lazy_stream -> 'a lazy_stream -> 'a lazy_stream =
  fun xs ys -> mux (fby (const true) (const false)) xs ys

(* Filtering a stream. Operator introduced in Lustre. Noted [whenc] here *)
(* because [when] is a keyword in OCaml *)
let rec whenc : 'a lazy_stream -> bool lazy_stream -> 'a lazy_stream =
  fun xs cs ->
  lazy
    (match Lazy.force xs, Lazy.force cs with
     | Cons(x, xs), Cons(c, cs) ->
        if c then Cons(x, whenc xs cs) else Lazy.force (whenc xs cs))

let when_not x c = whenc x (notl c)
                
(* Union of two streams. [merge] was originally introduced by Kahn *)
(* in its '74 paper. [merge] was a primitive of Lucid Synchrone [manual'06] *)
(* It replaces the operator [current] of Lustre because [current] cannot *)
(* be given a Kahn semantics without adding an extra clock argument *)
(* [merge] is a primitive of Scade introduced in version 6 in 2008); *)
(* it also replaced the [current] operator that does not exist anymore *)
let rec merge : bool lazy_stream -> 'a lazy_stream ->
                'a lazy_stream -> 'a lazy_stream =
  fun cs xs ys ->
  lazy
    (match Lazy.force cs with
     | Cons(true, cs) ->
        (match Lazy.force xs with
          Cons(x, xs) -> Cons(x, merge cs xs ys))
     | Cons(false, cs) ->
        (match Lazy.force ys with
           Cons(y, ys) -> Cons(y, merge cs xs ys)))


(* Least fix-point operator for streams *)
let fix: ('a lazy_stream -> 'a lazy_stream) -> 'a lazy_stream =
  fun f -> let rec y = lazy (Lazy.force (f y)) in
  y

let fix2:
      ('a lazy_stream * 'b lazy_stream ->
       'a lazy_stream * 'b lazy_stream) ->
          'a lazy_stream * 'b lazy_stream =
  fun f ->
  let xy = fix (fun xy -> pair (f (fst xy, snd xy))) in
  fst xy, snd xy

(* Conversion between streams and sequences *)
let rec nth: 'a lazy_stream -> (int -> 'a) =
  fun xs n ->
  match Lazy.force xs with
  | Cons(x, xs) -> if n = 0 then x else nth xs (n-1)

(* A representation of bottom (divergence) *)
(* [Lazy.force eps] raises exception CamlinternalLazy.Undefined. *)
let rec eps = lazy (Lazy.force eps)
let bot = eps
        
(* examples *)
(* 1. half *)
let half1 () =
  let f = fun x -> notl (pre false x) in
  fix f

let half2 () =
  let f = fun x -> (pre true (notl x)) in
  fix f

(* 2. natural numbers *)
let from v =
  let f = fun x -> pre v (plusl x (const 1)) in
  fix f

(*- integration: integr(x')(n) = sum_{i = 0}^n(x'(i)) *)
let integr: (int, int) system =
  fun x' -> fix (fun o -> plusl x' (pre 0 o))

let zero =
  let f = fun x -> pre 0 x in
  fix f

(* 2. unbounded memory. *)
(* run [out print_bool max_int (unbounded ())] *)
(* and see who allocated memory grows *)
let unbounded () =
  let t = const true in
  let h = half1 () in
  andl (whenc t h) t

(* 3. Causality loop *)
let deadlock1 () =
  let deadlock = fix (fun x -> x) in
  deadlock

let deadlock2 () =
  let deadlock = fix (fun x -> plusl x (const 1)) in
  deadlock

(* 3. N-synchrony [read e.g., paper at POPL'06] *)
(* This example cannot be writting in Lustre/Scade/Lucid Synchrone *)
(* yet, it executes in bounded memory (with a buffer of size 1 on one size *)
(* t          = 0 1 2 3 4 5 6 7 8 9  ...
 *-t when h   = 0   2   4   6   8 9  ...
 *-t whenot h =   1   3   5   7   9  ...
 *-result     =   1   5   9   13  17 ... *)
let n_synchrony () =
  let t = from 0 in
  let h = half1 () in
  plusl (whenc t (notl h)) (whenc t h)

let pair2 () =
  let nat = from 0 in
  pair (nat, whenc nat (half1 ()))
  
(* auxiliary functions *)
let rec list_of n xs =
  if n = 0 then []
  else match Lazy.force xs with
       | Cons(x, xs) -> x :: (list_of (n-1) xs)

let rec out print n xs =
  if n = 0 then ()
  else match Lazy.force xs with
       | Cons(x, xs) -> print x; print_string " . "; out print (n-1) xs

let print x = print_int x; print_string " "

let print_bool x = print_string (if x then "true" else "false")
;;

(* out print_bool 1000000000000 (unbounded ());; *)

(* Comments:
 *- This encoding of the Kahn semantics for process networks represents 
 *- streams as a lazy data-structures. 
 *- data-flow primitives are the unit delay [pre, fby], lifting for 
 *- constants and combinatorial functions [extend], and the filtering/combination
 *- functions [when/merge] from Lucid Synchrone. All the rest --- 
 *- function application, abstraction, fix-point are those of the host 
 *- language --- here OCaml.

 *- Strench of this encoding: 
 *- simple, a few lines of code;
 *- expressive: run both synchronous and non synchronous programs.
 *- but too much expressive/unconstrained: (1) non causal programs, that is
 *- programs that diverge.
 *- (1) is detected dynamically by OCaml runtime; it raises
 *- exception : CamlinternalLazy.Undefined.
 *- this is not satisfactory for real-time applications.
 *- (2) non synchronous may lead to unbounded buffering. this problem
 *-  is not detected. Try, e.g., out print max_int (unbounded ())
 *- and see how much memory is allocated to compute the constant true.
 *- this is not satisfactory for real-time applications!
 

 *- (1) This encoding is simple but it relies on lazy evaluation
 *- which is a complex evaluation mechanism. Moreover, it is difficult to answer
 *- questions like how much memory and time is necessary during an execution
 *- in particular Worst Case Execution Time (WCET).
 *- (2) Lazy evaluation is used here to define the operator [fix] that computes
 *- the lfp of a function.
 *- This operation is not total, that it may diverge or lead to a deadlock.
 *- It cannot be defined in Coq because Coq inposes guard conditions
 *- on recursive definitions to ensure that they define
 *- a value.
 *- (3) There is no way to represent bottom [bot] (divergence) as written above
 *- in Coq. We shall use a trick to represent it coinductively 
 *- (see kahn_epsilon.ml)
 *- (4) In a real-time context, we want to ensure at compile time that
 *- a program will never stuck; either for a deadlock (error 1) or because
 *- there is not enough memory (error 2); hence the purpose to define
 *- dedicated static analyses. Moreover, we want a run-time mechanism that
 *- is predictive in term of time and space; hence an dedicated/different
 *- implementation for streams. *)
