(* An encoding of the Lustre/Lucid Synchrone data-flow operators in OCaml *)
(* MP - oct. 2021 - MPRI - Course notes *)

(* control structures; weak and strong preemption; termination *)
(* This is based on an old work by Caspi on recursive block diagram *)
(* in 1993; Caspi and Pouzet in 1994-1997; by Pouzet (habilitation, 2002) *)
(* The main intuition is this: the modular reset and the so-called *)
(* control structures of Esterel can be represented by recursive stream *)
(* functions. The kind of recursion that is used is similar to *)
(* classical tail-recursion: when a recursive stream function is called *)
(* the calling context is no more useful. *)

(* e ::= reset e every e | do e until e then e | do e until e | exit A
         | catch e with A -> e1 | ... A -> en
*)

open Kahn_lazy
   
(* A particular form of [merge]. When the condition is true, do not *)
(* use the first branch anymore              *)
(* cs = 0  0  0  0  0  1  0  1 1 0 0 0 0 1 0 *)
(* xs = x0 x1 x2 x3 x4 x5 x6 ...             *)
(* ys =                   y0 y1 y2 y3 ...    *)
(* zs = x0 x1 x2 x3 x4 x5 y0 y1 y2 y3 ...    *)
(* merge_until false.cs x.xs ys = x.merge_until cs xs ys *)
(* merge_until true.cs x.xs ys = x.ys *)
let rec merge_until : bool lazy_stream -> 'a lazy_stream ->
                     'a lazy_stream -> 'a lazy_stream =
  fun cs xs ys ->
  lazy
    (match Lazy.force cs with
     | Cons(c, cs) ->
        (match Lazy.force xs with
         | Cons(x, xs) ->
            Cons(x, if c then ys else merge_until cs xs ys)))

(* The same for filtering.                      *)
(* cs = 0  0  0  0  0  1  0  1  1  0  0 0 0 1 0 *)
(* xs = x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 ...       *)
(* ys =                   x6 x7 x8 x9 ...       *)
(* after x.xs true.cs = xs *)
(* after x.xs false.cs = after xs cs *)
let rec after_until : 'a lazy_stream -> bool lazy_stream -> 'a lazy_stream =
  fun xs cs ->
  lazy
    (match Lazy.force xs, Lazy.force cs with
     | Cons(x, xs), Cons(c, cs) ->
        Lazy.force (if c then xs else after_until xs cs))
  
(* cs = 0  0  0  0  0  1  0  1 1 0 0 0 0 1 0 *)
(* xs = x0 x1 x2 x3 x4 ... *)
(* ys =                y0 y1 y2 y3 ... *)
(* zs = x0 x1 x2 x3 x4 y0 y1 y2 y3 ... *)
let rec merge_unless : bool lazy_stream -> 'a lazy_stream ->
                     'a lazy_stream -> 'a lazy_stream =
  fun cs xs ys ->
  match Lazy.force cs with
  | Cons(c, cs) ->
        if c then ys
        else
          match Lazy.force xs with
          | Cons(x, xs) ->
             lazy (Cons(x, merge_unless cs xs ys))

(* after_unless. *)
(* cs = 0  0  0  0  0  1  0  1  1  0  0 0 0 1 0 *)
(* xs = x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 ... *)
(* ys =                x5 x6 x7 x8 ... *)
let rec after_unless : 'a lazy_stream -> bool lazy_stream -> 'a lazy_stream =
  fun xs cs ->
  lazy
    (match Lazy.force cs with
     | Cons(c, cs) ->
        Lazy.force (if c then xs else after_unless xs cs))

(* init *)
(* cs            = 1 0 0 0 1 1 1 1 ... *)
(* false_true cs = 0 0 0 0 1 1 1 1 ... *)
let false_true : bool lazy_stream -> bool lazy_stream =
  fun cs ->
  lazy
    (match Lazy.force cs with
     | Cons(_, cs) -> Cons(false, cs))

(* while_not *)
(* xs =     0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 1 1 1 0 ... *)
(* ys =     1 1 1 1 1 1 1 0 0 0 0 0 0 0 0 0 0 0 0 ... *)
(* not ys = 0 0 0 0 0 0 0 1 1 1 1 1 1 1 1 1 1 1 1 ... *)
let while_not x =
  (* let rec o = (true fby o) && not x in o *)
  fix (fun o -> andl (pre true o) (notl x))

let do_unless: ('a, bool) system -> ('a, 'b) system ->
    ('a, 'b) system -> ('a, 'b) system =
  fun c f g x ->
    let c = while_not (c x) in
    merge c (f (whenc x c)) (g (whenc x (notl c)))

(*
let reset f c x =
  let rec reset x =
    do_unless (fun x -> false_true (c x)) f reset x in
  reset x
 *)

 
(* Activation condition. Run [f] and [g]; [f] sees the sub-stream of [x] *)
(* when [c] is true; [g] the complementary stream (when [c] is false) *)
let activate:
      bool lazy_stream -> ('a, 'b) system ->
      ('a, 'b) system -> ('a, 'b) system =
  fun c f g x ->
    merge c (f (whenc x c)) (g (when_not x c))

(* Sequence. Runs [f] while [c] is false; then [g] *)
(* [f], [c] and [g] stands for systems (block diagrams), *)
(* hence stream functions *)
let do_until:
      ('a, bool) system -> ('a, 'b) system ->
      ('a, 'b) system -> ('a, 'b) system =
  fun c f g x ->
    let c = c x in
    merge_until c (f x) (g (after_until x c))

(* Sequence. Runs [f] while [c] is false; then [g] *)
(* [f], [c] and [g] stands for systems (block diagrams), *)
(* hence stream functions *)
let do_unless:
      ('a, bool) system -> ('a, 'b) system ->
      ('a, 'b) system -> ('a, 'b) system =
  fun c f g x ->
    let c = c x in
    merge_unless c (f x) (g (after_unless x c))

(* Reset. [reset f c x]. Runs [f] but resets it every time [c] is true *)
(* [f], [c] and [g] stands for systems (block diagrams), hence *)
(* stream functions *)
(* [reset] is defined recursively. *)
(*
let reset f c x =
  let rec reset x =
    do_until c f reset x in
  reset x


let reset f c x =
  let rec reset x =
    do_unless (fun x -> false_true (c x)) f reset x in
  reset x
 *)

(* An alternative encoding of the reset - *)
(* Not expressed here as a sequence such that:
 *- [reset f every c = do f unless (fun x -> (false -> c x)) then reset f every c]
 *)
(* original idea by Paul Jeanmaire, Oct. 2021 *)
let hd xs = match Lazy.force xs with | Cons(x, _) -> x
let tl xs = match Lazy.force xs with | Cons(_, xs) -> xs
let hd_tl xs = match Lazy.force xs with | Cons(x, xs) -> x, xs
                                                       
let reset f rs xs =
  let rec reset_aux ys rs xs =
    lazy
      (match Lazy.force rs with
         Cons(r, rs) ->
         if r then
           Lazy.force (reset_aux (f xs) rs (tl xs))
         else
           let y, ys = hd_tl ys in
           Cons(y, reset_aux ys rs (tl xs))) in
  reset_aux (f xs) rs xs

let reset2 f rs xs1 xs2 =
  let ys = reset f rs (pair (xs1, xs2)) in
  fst ys, snd ys

let cond xy =
  let x = fst xy in
  let y = snd xy in
  let o = merge x y (const 0) in
  o

let reset_cond rs xs = reset cond rs xs
                     
(* Examples *)

(* 1. sequence of two behaviors *)
let from : (int, int) system =
  fun n -> fix (fun x -> fby (const 0) (plusl x (const 1)))

let cond : (int, bool) system =
  fun x -> eql (from (const 0)) (const 42)
           
let r = list_of 100
          (do_unless cond from
             (do_unless cond from
                from)
             (const 0))

let r = list_of 100
          (do_until (fun _ -> const false) from (fun _ -> eps) (const 0))

let p () =
  let p = fix (fun p -> pre true (pre true (pre true (notl p)))) in
  p
  
let r =
  list_of 100 (reset from (p ()) (from (const 0)))
      
let r = list_of 100
          (do_unless
             (fun _ -> const false) from
             (fun _ -> eps) (const 0))

let r = list_of 100
          (do_unless
             (fun _ -> const false) from
             (do_unless
                (fun _ -> const false) from
                (fun _ -> eps)) (const 0))

let r = list_of 100
          (reset
             from
             (fun _ -> const false)
             (eps))

(* 2. Reset an integrator at step [n] *)
(*- integration: integr(x')(n) = sum_{i = 0}^n(x'(i)) *)
let integr: (int, int) system =
  fun x' -> fix (fun o -> plusl x' (pre 0 o))

let integr2 x'' = integr (integr x'')

let integr_reset m = reset integr2 cond (const 1)

(* An automaton with two modes. Up and down. *)
(*
 *- This system is written the following way in Zelus.
 *- let node two_modes v =
 *- automaton
 *- | Incr -> do o = last o + 1 until (o >= v) then Decr
 *- | Decr -> do o = last o - 1 until (o <= -v) then Incr
 *- end

*-  We write it as two mutually recursive stream functions incr and decr:

*-  let rec up x =
*-    do_until (greater 42) incr down
*-  and down x =
*-    do_until (lesser (-42) decr up
*)       

