function [Ks_train,Ks_test,kernel_types] = create_path_kernel_matrices(datatrain,datatest,kernelparams)

% Creates common kernel matrices
% NB : all kernel matrices are normalized to unit trace
%
% [Ks_train,Ks_test,kernel_types] = create_path_kernel_matrices(datatrain,datatest,kernelparams)
%
% INPUT
% In the following, n and m are the training set and test set sizes, and
% d is the dimensionality of the data
%     datatrain : d*n data
%      datatest : d*m data
%  kernelparams : a struct with the following fields
%             LINEAR : {0,1} include the linear kernel on all variables?
%        FULL_LINEAR : {0,1} include linear kernels on each variable separately?
%           CONSTANT : {0,1} include the constant kernel?
%           IDENTITY : {0,1} include the identity matrix?
%         POLYNOMIAL : {0,1} include polynomial kernel?
%      FULL_GAUSSIAN : {0,1} include Gaussian kernels on all variables?
%    SUBSET_GAUSSIAN : {0,1} include Gaussian kernels on subsets of variables?
%            NSIGMAS : {integer} number of width parameters for Gaussian kernels
%         MAXSUBSETS : {integer} maximal size of subsets for Gaussian kernels
%      POLY_MAXORDER : {integer} maximal order of polynomial kernel
%          SIGMAGRID : {real} spacing between values of the log width for Gaussian kernels
%
% OUTPUT
% In the following, k is the number of kernels
%      Ks_train : 1*k cell array of n*n kernel matrices
%       Ks_test : 1*k cell array of n*m kernel matrices
%  kernel_types : 1*k cell array of strings containing the kernel name


% TYPES OF KERNEL
LINEAR          = kernelparams.LINEAR ;
FULL_LINEAR     = kernelparams.FULL_LINEAR ;
CONSTANT        = kernelparams.CONSTANT ;
IDENTITY        = kernelparams.IDENTITY ;
POLYNOMIAL      = kernelparams.POLYNOMIAL ;
FULL_GAUSSIAN   = kernelparams.FULL_GAUSSIAN ;
SUBSET_GAUSSIAN = kernelparams.SUBSET_GAUSSIAN ;
NSIGMAS         = kernelparams.NSIGMAS ;
MAXSUBSETS      = kernelparams.MAXSUBSETS ;
POLY_MAXORDER   = kernelparams.POLY_MAXORDER ;
SIGMAGRID       = kernelparams.SIGMAGRID;
datatrain = datatrain';
datatest = datatest';

[d, ntrain] = size(datatrain); 
[d, ntest]  = size(datatest);

sigmas  = 10.^ ( SIGMAGRID * (-NSIGMAS+1:2:NSIGMAS) );


% build Gaussian kernel matrices on all
k=1;
if FULL_GAUSSIAN
    for j=1:NSIGMAS
        Ks_train{k} = exp( - sigmas(j) * sqdist(datatrain, datatrain ) / d )/ntrain;
        Ks_test{k} = exp( - sigmas(j) * sqdist(datatrain, datatest ) / d )/ntrain;
        kernel_types.type{k} = 'gaussian';
        kernel_types.params1{k} = 1:d;
        kernel_types.params2{k} = sigmas(j);
        k=k+1;
    end
end

% build Gaussian kernel matrices on singletons and subsets
if SUBSET_GAUSSIAN
    k=1;
    for i=1:d
        for j=1:NSIGMAS
            Ks_train{k} = exp( - sigmas(j) * sqdist(datatrain(i,:), datatrain(i,:) ) )/ntrain;
            Ks_test{k} = exp( - sigmas(j) * sqdist(datatrain(i,:), datatest(i,:) ) )/ntrain;
            kernel_types.type{k} = 'gaussian';
            kernel_types.params1{k} = i;
            kernel_types.params2{k} = sigmas(j);
            k=k+1;
        end
    end
    
    
    
    if MAXSUBSETS>1
        % build kernel matrices on pairs
        for i=2:d
            for i2=1:i-1
                for j=1:NSIGMAS
                    Ks_train{k} = exp( - sigmas(j) * sqdist(datatrain([i i2],:), datatrain([i i2],:) ) )/ntrain;
                    Ks_test{k} = exp( - sigmas(j) * sqdist(datatrain([i i2],:), datatest([i i2],:) ) )/ntrain;
                    kernel_types.type{k} = 'gaussian';
                    kernel_types.params1{k} = [i i2];
                    kernel_types.params2{k} = sigmas(j);
                    
                    k=k+1;
                end
            end
        end
        
    end
end

if LINEAR
    % add linear kernel matrices
    for i=1:d
        Ks_train{k} = datatrain(i,:)'*datatrain(i,:) ;
        Ks_test{k} = datatrain(i,:)'*datatest(i,:) ;
        if trace(Ks_train{k}) > 10^-30
            Ks_test{k} = Ks_test{k} / trace(Ks_train{k});
            Ks_train{k} = Ks_train{k} / trace(Ks_train{k});
        end
        kernel_types.type{k} = 'linear';
        kernel_types.params1{k} = i;
        
        k=k+1;
    end
end

if FULL_LINEAR
    % add linear kernel matrices
    Ks_train{k} = datatrain'*datatrain;
    Ks_test{k} = datatrain'*datatest;
    Ks_test{k} = Ks_test{k} / trace(Ks_train{k});
    Ks_train{k} = Ks_train{k} / trace(Ks_train{k});
    kernel_types.type{k} = 'linear';
    kernel_types.params1{k} = 1:d;
    k=k+1;
end



if IDENTITY
    % add identity matrix
    Ks_train{k} = eye( ntrain)/ntrain;
    Ks_test{k} = zeros(ntrain,ntest)/ntrain;
    kernel_types.type{k} = 'identity';
    k = k + 1;
end

if CONSTANT
    % add constant matrix
    Ks_train{k} = ones( ntrain)/ntrain;
    Ks_test{k} = ones(ntrain,ntest)/ntrain;
    kernel_types.type{k} = 'constant';
    k = k + 1;
end





if POLYNOMIAL
    for i=2:POLY_MAXORDER
        % add linear kernel matrices
        Ks_train{k} = datatrain'*datatrain / d ;
        maxK = max(abs(Ks_train{k}(:)));
        Ks_train{k} = ( 1 + .5 * Ks_train{k}/maxK ).^i;
        Ks_test{k} = datatrain'*datatest / d;
        Ks_test{k} = ( 1 + .5 * Ks_test{k}/maxK ).^i;
        
        Ks_test{k} = Ks_test{k} / trace(Ks_train{k});
        Ks_train{k} = Ks_train{k} / trace(Ks_train{k});
        kernel_types.type{k} = 'polynomial';
        kernel_types.params1{k} = i;
        k=k+1;
    end
end


